<?php
/*
 * Copyright REZO ZERO 2015
 *
 * IssueFeed
 *
 * @file IssueFeedController.php
 * @copyright REZO ZERO 2015
 * @author Thomas Aufresne
 */
namespace Themes\KiosqueTheme\Controllers\Blocks;

use Doctrine\ORM\NoResultException;
use RZ\Roadiz\Core\Entities\Node;
use RZ\Roadiz\Core\Entities\NodesSources;
use RZ\Roadiz\Core\Entities\Translation;
use Symfony\Component\HttpFoundation\Request;
use Themes\KiosqueTheme\KiosqueThemeApp;

/**
 * IssueFeedController class
 */
class IssueFeedController extends KiosqueThemeApp
{
    const ITEM_PER_PAGE = 12;

    /**
     * {@inheritdoc}
     */
    public function blockAction(Request $request, NodesSources $source, $assignation)
    {
        $this->prepareNodeSourceAssignation($source, $source->getTranslation());
        $this->assignation = array_merge($this->assignation, $assignation);
        $this->assignation['issueTypes'] = $this->getAvailableTypes();
        $this->assignation['issueTags'] = $this->getAvailableTags();

        $listManager = $this->getEntityListManager($request);
        $this->assignation['issues'] = $listManager->getEntities();
        $this->assignation['filters'] = $listManager->getAssignation();

        return $this->render('render-blocks/issuefeed.html.twig', $this->assignation);
    }

    protected function getAvailableTypes()
    {
        try {
            $qb = $this->getService('em')
                ->getRepository('GeneratedNodeSources\\NSIssue')
                ->createQueryBuilder('i');

            $qb->select('i.issuetype')
                ->innerJoin('i.node', 'n')
                ->andWhere($qb->expr()->isNotNull('i.issuetype'))
                ->andWhere($qb->expr()->lte('i.issuedate', ':now'))
                ->andWhere($qb->expr()->eq('i.translation', ':translation'))
                ->andWhere($qb->expr()->eq('n.status', ':status'))
                ->setParameter(':now', new \Datetime('now'))
                ->setParameter(':translation', $this->translation)
                ->setParameter(':status', Node::PUBLISHED)
                ->distinct();

            return array_map('current', $qb->getQuery()->getScalarResult());
        } catch (NoResultException $e) {
            return [];
        }
    }

    protected function getAvailableTags()
    {
        try {
            $qb = $this->getService('em')
                ->getRepository('RZ\Roadiz\Core\Entities\Tag')
                ->createQueryBuilder('t');

            $qb->select('t')
                ->innerJoin('t.nodes', 'n')
                ->innerJoin('t.translatedTags', 'tt')
                ->andWhere($qb->expr()->eq('tt.translation', ':translation'))
                ->andWhere($qb->expr()->eq('t.parent', ':parent'))
                ->andWhere($qb->expr()->eq('n.nodeType', ':nodeType'))
                ->andWhere($qb->expr()->eq('n.status', ':status'))
                ->setParameter(':status', Node::PUBLISHED)
                ->setParameter(':parent', $this->themeContainer['themesTag'])
                ->setParameter(':nodeType', $this->themeContainer['typeIssue'])
                ->setParameter(':translation', $this->translation)
                ->distinct();

            return $qb->getQuery()->getResult();
        } catch (NoResultException $e) {
            return [];
        }
    }

    /**
     * Construct entityListManager for issue.
     *
     * @param  Request $request
     * @return EntityListManager
     */
    public function getEntityListManager(Request $request)
    {
        $criteria = [
            'node.parent' => $this->node,
            'node.nodeType' => $this->themeContainer['typeIssue'],
            'translation' => $this->translation,
            'issuedate' => ['<=', new \Datetime('now')],
        ];
        $ordering = [
            'sticky' => 'DESC',
            'issuedate' => 'DESC',
        ];

        if ($request->query->has('type')) {
            $criteria['issuetype'] = $request->query->get('type');
        }

        if ($request->query->has('tag')) {
            $tagsNames = explode('+', $request->query->get('tag'));
            $tags = $this->getService('tagApi')->getBy(['tagName' => $tagsNames]);

            if (null !== $tags) {
                $criteria['tags'] = $tags;
            }
        }

        $listManager = $this->createEntityListManager(
            'GeneratedNodeSources\\NSIssue',
            $criteria,
            $ordering
        );

        /*
         * First, set item per page
         */
        $itemPerPage = $this->nodeSource->getItemsPerPage() > 0 ? $this->nodeSource->getItemsPerPage() : static::ITEM_PER_PAGE;
        $listManager->setItemPerPage($itemPerPage);
        /*
         * Second, handle the manager
         */
        $listManager->handle();

        return $listManager;
    }

    public function searchAction(Request $request, $terms, $_locale = 'fr')
    {

        $translation = $this->bindLocaleFromRoute($request, $_locale);

        $this->prepareThemeAssignation($this->getHome(), $translation);

        $this->assignation['results'] = $this->getService('nodeSourceApi')->getBy(
            [
                'node.nodeType' => $this->themeContainer['typeIssue'],
                'translation' => $this->translation,
                'title' => ['LIKE', '%' . $terms . '%'],
            ]
        );

        return $this->render('types/search.html.twig', $this->assignation);
    }
}
